# 画面設計書 7-File Encryption

## 概要

本ドキュメントは、VBCorLibプロジェクトのサンプルアプリケーション「File Encryption」の画面設計書です。

### 本画面の処理概要

File Encryptionは、VBCorLibライブラリが提供するSymmetricAlgorithmクラス群とCryptoStreamを使用して、ファイルの暗号化/復号機能をデモンストレーションする画面です。DES、TripleDES、RC2、Rijndael（AES）の4種類の対称暗号アルゴリズムに対応し、鍵サイズ、ブロックサイズ、暗号化モード、パディングモードを選択できます。

**業務上の目的・背景**：ファイルの暗号化は、機密データの保護において重要な機能です。本画面は、VBCorLibが提供する対称暗号アルゴリズムの使用方法を包括的にデモンストレーションし、開発者が適切なアルゴリズムとパラメータを選択して暗号化機能を実装できるよう支援することを目的としています。Rfc2898DeriveBytesによるパスワードベースの鍵導出も示しています。

**画面へのアクセス方法**：Examples/Cryptography/FileEncryption/FileEncryption.vbpプロジェクトを開き、F5キーで実行することでこの画面が表示されます。

**主要な操作・処理内容**：
1. アルゴリズムの選択（DES、TripleDES、RC2、Rijndael）
2. 鍵サイズ、ブロックサイズ、暗号化モード、パディングの設定
3. パスワードの入力
4. 入力ファイルの選択（Browse）
5. 「Encrypt」ボタンで暗号化実行
6. 「Decrypt」ボタンで復号実行

**画面遷移**：本画面は独立したデモアプリケーションであり、他の画面への遷移はありません。

**権限による表示制御**：本画面は開発者向けデモツールであり、特別な権限制御は実装されていません。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 70 | CryptoStream | 主機能 | ファイル暗号化/復号ストリーム（暗号化/復号化処理の実行） |
| 49 | RijndaelManaged | 主機能 | Rijndael暗号化アルゴリズム（選択可能な対称暗号の1つ） |
| 50 | DESCryptoServiceProvider | 主機能 | DES暗号化アルゴリズム（選択可能な対称暗号の1つ） |
| 51 | TripleDESCryptoServiceProvider | 主機能 | Triple DES暗号化アルゴリズム（選択可能な対称暗号の1つ） |
| 52 | RC2CryptoServiceProvider | 主機能 | RC2暗号化アルゴリズム（選択可能な対称暗号の1つ） |
| 69 | Rfc2898DeriveBytes | 主機能 | パスワードから暗号鍵を導出（PBKDF2アルゴリズム） |
| 34 | File | 補助機能 | ファイルオープン操作（OpenFile） |
| 1 | CorArray | 補助機能 | IV配列の作成（CreateInstance） |

## 画面種別

ファイル処理 / 暗号化デモンストレーション

## URL/ルーティング

VB6デスクトップアプリケーションのため、URLルーティングは存在しません。フォームファイル: `Examples/Cryptography/FileEncryption/Form1.frm`

## 入出力項目

| 項目名 | コントロール | 項目種別 | データ型 | 説明 |
|--------|-------------|----------|----------|------|
| cboAlgorithms | ComboBox | 入力 | String | 暗号化アルゴリズム選択 |
| cboKeySizes | ComboBox | 入力 | Long | 鍵サイズ選択（bits） |
| cboBlockSizes | ComboBox | 入力 | Long | ブロックサイズ選択（bits） |
| cboMode | ComboBox | 入力 | Long | 暗号化モード選択（CBC、ECB等） |
| cboPadding | ComboBox | 入力 | Long | パディングモード選択 |
| txtPassword | TextBox | 入力 | String | パスワード |
| txtInputFileName | TextBox | 入力 | String | 入力ファイルパス |
| txtOutputFileName | TextBox | 入力 | String | 出力ファイルパス |

## 表示項目

| 項目名 | コントロール | データ型 | 説明 |
|--------|-------------|----------|------|
| pbrProgress | ProgressBar | Long | 処理進捗表示 |
| Label1-8 | Label | String | 各入力フィールドのラベル |

## イベント仕様

### 1-Form_Load

フォーム読み込み時に初期設定を行います。

**処理内容**:
1. cboPaddingのListIndexを1に設定（PKCS7）
2. cboModeのListIndexを0に設定（CBC）

**コード参照**: 行271-274

### 2-cboAlgorithms_Click

アルゴリズム選択時にCspを初期化し、関連ComboBoxを更新します。

**処理内容**:
1. CryptoConfig.CreateFromNameでSymmetricAlgorithmを生成
2. FillKeySizesComboBoxで鍵サイズ選択肢を更新
3. FillBlockSizesComboBoxでブロックサイズ選択肢を更新

**コード参照**: 行218-225

### 3-cmdEncrypt_Click

ファイルを暗号化します。

**処理内容**:
1. PrepareCspでCspを設定
2. mCsp.CreateEncryptorでEncryptorを取得
3. DoCipherで暗号化処理実行

**コード参照**: 行255-269

### 4-cmdDecrypt_Click

ファイルを復号します。

**処理内容**:
1. PrepareCspでCspを設定
2. mCsp.CreateDecryptorでDecryptorを取得
3. DoCipherで復号処理実行

**コード参照**: 行239-253

### 5-PrepareCsp（内部関数）

暗号化パラメータを設定します。

**処理内容**:
1. Mode、Padding、KeySize、BlockSizeを設定
2. GenerateKeyでパスワードから鍵を導出
3. CorArray.CreateInstanceでIVを生成（ゼロ埋め）

**コード参照**: 行300-311

```vb
Private Sub PrepareCsp()
    mCsp.Mode = cboMode.ListIndex + 1
    mCsp.Padding = cboPadding.ListIndex + 1
    mCsp.KeySize = cboKeySizes.ItemData(cboKeySizes.ListIndex)
    mCsp.BlockSize = cboBlockSizes.ItemData(cboBlockSizes.ListIndex)
    mCsp.Key = GenerateKey(txtPassword.Text, mCsp.KeySize \ 8)
    mCsp.IV = CorArray.CreateInstance(vbByte, mCsp.BlockSize \ 8)
End Sub
```

### 6-DoCipher（内部関数）

暗号化/復号処理を実行します。

**処理内容**:
1. 入力ファイルをStreamで開く
2. CryptoStreamで出力ファイルを開く
3. ProcessFileでバッファ単位で処理

**コード参照**: 行280-295

### 7-GenerateKey（内部関数）

パスワードから暗号鍵を生成します。

**処理内容**:
1. ソルト配列を作成（ゼロ埋め）
2. Rfc2898DeriveBytesでPBKDF2鍵導出
3. GetBytesで必要サイズの鍵を取得

**コード参照**: 行389-402

```vb
Private Function GenerateKey(ByVal Password As String, ByVal KeySize As Long) As Byte()
    Dim Salt() As Byte
    ReDim Salt(0 To KeySize - 1)

    Dim Generator As Rfc2898DeriveBytes
    Set Generator = Cor.NewRfc2898DeriveBytes(Password, Salt)

    GenerateKey = Generator.GetBytes(KeySize)
End Function
```

## データベース更新仕様

本画面はデータベースを使用しません。

## メッセージ仕様

| メッセージID | メッセージ内容 | 表示条件 |
|-------------|---------------|----------|
| MSG-01 | Exception.ToString | 暗号化/復号エラー発生時 |

## 例外処理

| 例外種別 | 対処方法 |
|---------|---------|
| 暗号化エラー | Catch(Ex, Err)でExceptionをキャッチしMsgBox表示 |
| 復号エラー | Catch(Ex, Err)でExceptionをキャッチしMsgBox表示 |
| ファイルエラー | VBランタイムエラー発生 |

## 備考

- BorderStyle = Fixed Dialog（サイズ変更不可）
- ShowInTaskbar = False
- COMDLG32.OCX（CommonDialog）を使用
- MSCOMCTL.OCX（ProgressBar）を使用
- 4Kバッファで処理（ProcessFile内でReDim b(4095)）
- IVはゼロ埋め（シンプルさを優先）

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

暗号化処理で使用されるオブジェクトを理解します。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Form1.frm | `Examples/Cryptography/FileEncryption/Form1.frm` | 行206-214: コメント説明とmCsp変数宣言 |

**読解のコツ**: mCspはSymmetricAlgorithmインターフェースとして宣言されており、実行時にDES、TripleDES、RC2、Rijndaelのいずれかが代入されます。

```vb
Private mCsp As SymmetricAlgorithm
```

#### Step 2: エントリーポイントを理解する

アルゴリズム選択時の初期化がエントリーポイントです。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Form1.frm | `Examples/Cryptography/FileEncryption/Form1.frm` | 行218-225: cboAlgorithms_Click |

**主要処理フロー**:
1. **行220**: CryptoConfig.CreateFromNameでアルゴリズムを生成
2. **行223-224**: 鍵サイズ・ブロックサイズの選択肢を更新

#### Step 3: 暗号化処理フローを理解する

暗号化/復号の実行処理を確認します。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Form1.frm | `Examples/Cryptography/FileEncryption/Form1.frm` | 行300-311: PrepareCsp |
| 3-2 | Form1.frm | `Examples/Cryptography/FileEncryption/Form1.frm` | 行280-295: DoCipher |
| 3-3 | Form1.frm | `Examples/Cryptography/FileEncryption/Form1.frm` | 行320-345: ProcessFile |

**主要処理フロー（暗号化）**:
- **行301-304**: Mode, Padding, KeySize, BlockSizeを設定
- **行307**: パスワードから鍵を導出
- **行310**: IVを生成
- **行283**: File.OpenFileで入力ストリーム
- **行289**: NewCryptoStream(出力, Transform, WriteMode)
- **行330-344**: 4Kバッファ単位で読み込み→暗号化→書き込み

#### Step 4: 鍵導出処理を理解する

パスワードから暗号鍵を生成する処理を確認します。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Form1.frm | `Examples/Cryptography/FileEncryption/Form1.frm` | 行389-402: GenerateKey |

**主要処理フロー（鍵導出）**:
- **行392**: ソルト配列作成（KeySizeバイトのゼロ埋め）
- **行397-398**: Rfc2898DeriveBytesでPBKDF2インスタンス作成
- **行402**: GetBytesで指定サイズの鍵を取得

### プログラム呼び出し階層図

```
cboAlgorithms_Click
    │
    ├─ CryptoConfig.CreateFromName
    │      └─ mCsp = SymmetricAlgorithm
    │
    ├─ FillKeySizesComboBox
    │      └─ FillSizes(mCsp.LegalKeySizes, cboKeySizes)
    │
    └─ FillBlockSizesComboBox
           └─ FillSizes(mCsp.LegalBlockSizes, cboBlockSizes)

cmdEncrypt_Click / cmdDecrypt_Click
    │
    ├─ PrepareCsp
    │      ├─ mCsp.Mode, Padding, KeySize, BlockSize設定
    │      ├─ GenerateKey(Password, KeySize)
    │      │      └─ Rfc2898DeriveBytes → GetBytes
    │      └─ CorArray.CreateInstance(vbByte, BlockSize)
    │
    └─ DoCipher(mCsp.CreateEncryptor/Decryptor)
           ├─ File.OpenFile(入力)
           ├─ NewCryptoStream(File.OpenFile(出力), Transform, WriteMode)
           └─ ProcessFile(入力Stream, 出力CryptoStream)
                  └─ 4Kバッファループ
                         ├─ Src.ReadBlock
                         └─ Dst.WriteBlock
```

### データフロー図

```
[ユーザー入力]
アルゴリズム選択 ───▶ CryptoConfig.CreateFromName ───▶ mCsp
パスワード ─────────▶ Rfc2898DeriveBytes ──────────────▶ 暗号鍵

[暗号化処理]
入力ファイル ───▶ FileStream ───▶ ReadBlock(4K) ───▶ CryptoStream ───▶ 出力ファイル
                                                    (WriteMode)

[復号処理]
入力ファイル ───▶ FileStream ───▶ ReadBlock(4K) ───▶ CryptoStream ───▶ 出力ファイル
                                                    (WriteMode+Decryptor)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Form1.frm | `Examples/Cryptography/FileEncryption/Form1.frm` | ソース | メインフォーム |
| Form1.frx | `Examples/Cryptography/FileEncryption/Form1.frx` | バイナリ | フォームリソース |
| FileEncryption.vbp | `Examples/Cryptography/FileEncryption/FileEncryption.vbp` | プロジェクト | プロジェクト定義 |
| COMDLG32.OCX | 外部 | OCX | CommonDialogコントロール |
| MSCOMCTL.OCX | 外部 | OCX | ProgressBarコントロール |
| VBCorLib.dll | 外部 | DLL | VBCorLibライブラリ本体 |
